<?php
/**
 * Plugin Name: Balor — Quiz (UI Builder)
 * Description: Onedio tarzı A/B/C/D çoğunluk testlerini JSON girmeden oluşturun. Sonuçlara görsel ekleyin. Kısa kod: [balor_quiz id="POST_ID"]
 * Version: 1.2.0
 * Author: Balor
 * License: GPLv2 or later
 * Text Domain: balor-quiz
 */

if (!defined('ABSPATH')) exit;

define('BQUI_VER','1.2.0');
define('BQUI_DIR', plugin_dir_path(__FILE__));
define('BQUI_URL', plugin_dir_url(__FILE__));

// Register CPT
add_action('init', function(){
  register_post_type('balor_quiz_cpt',[
    'labels'=>[
      'name'=>__('Balor Quizzes','balor-quiz'),
      'singular_name'=>__('Balor Quiz','balor-quiz'),
      'menu_name'=>__('Balor','balor-quiz'),
      'add_new_item'=>__('Yeni Quiz Ekle','balor-quiz'),
      'edit_item'=>__('Quizi Düzenle','balor-quiz'),
    ],
    'public'=>true,
    'show_in_rest'=>false,
    'menu_icon'=>'dashicons-awards',
    'supports'=>['title','thumbnail','excerpt'],
    'rewrite'=>['slug'=>'balor-quiz']
  ]);
});

// Admin assets
add_action('admin_enqueue_scripts', function($hook){
  global $post;
  if( ($hook==='post-new.php' || $hook==='post.php') && isset($post->post_type) && $post->post_type==='balor_quiz_cpt' ){
    if(function_exists('wp_enqueue_media')) wp_enqueue_media();
    wp_enqueue_style('bqui-admin', BQUI_URL.'assets/css/admin.css',[],BQUI_VER);
    wp_enqueue_script('bqui-admin', BQUI_URL.'assets/js/admin.js',['jquery'],BQUI_VER,true);
  }
});

// Data helpers
function hqui_get_data($post_id){
  $data = get_post_meta($post_id, '_hqui_data', true);
  $arr = is_string($data) ? json_decode($data,true) : (is_array($data)?$data:[]);
  if(!is_array($arr)) $arr = [];
  $arr += [
    'intro_title' => '',
    'intro_desc' => '',
    'tie_break_order' => ['A','B','C','D'],
    'questions' => [],
    'results' => [
      'A'=>['title'=>'','desc'=>'','image_id'=>0],
      'B'=>['title'=>'','desc'=>'','image_id'=>0],
      'C'=>['title'=>'','desc'=>'','image_id'=>0],
      'D'=>['title'=>'','desc'=>'','image_id'=>0],
    ],
  ];
  return $arr;
}

// Builder metabox
add_action('add_meta_boxes', function(){
  add_meta_box('hqui_builder', __('Quiz Builder','balor-quiz'), 'hqui_render_builder', 'balor_quiz_cpt', 'normal', 'high');
});

function hqui_render_builder($post){
  wp_nonce_field('hqui_save','hqui_nonce');
  $d = hqui_get_data($post->ID); ?>
  <div class="hqui">
    <div class="hqui-row">
      <label><?php _e('Giriş Başlığı','balor-quiz');?></label>
      <input type="text" name="hqui[intro_title]" value="<?php echo esc_attr($d['intro_title']);?>" class="widefat">
    </div>
    <div class="hqui-row">
      <label><?php _e('Giriş Açıklaması','balor-quiz');?></label>
      <textarea name="hqui[intro_desc]" class="widefat" rows="3"><?php echo esc_textarea($d['intro_desc']);?></textarea>
    </div>
    <div class="hqui-row">
      <label><?php _e('Eşitlikte Öncelik Sırası (A,B,C,D)','balor-quiz');?></label>
      <input type="text" name="hqui[tie_break]" value="<?php echo esc_attr(implode(',', $d['tie_break_order']));?>" class="regular-text">
      <p class="description"><?php _e('Örn: A,B,C,D','balor-quiz');?></p>
    </div>

    <hr>
    <h3><?php _e('Sorular','balor-quiz');?></h3>
    <div id="hqui-questions">
      <?php if(!empty($d['questions'])): foreach($d['questions'] as $qi=>$q): hqui_question_block($qi, $q); endforeach; endif; ?>
    </div>
    <p><button type="button" class="button button-primary" id="hqui-add-q"><?php _e('Soru Ekle','balor-quiz');?></button></p>

    <hr>
    <h3><?php _e('Sonuçlar (A/B/C/D)','balor-quiz');?></h3>
    <div class="hqui-results-grid">
      <?php foreach(['A','B','C','D'] as $k): $r = isset($d['results'][$k]) ? $d['results'][$k] : ['title'=>'','desc'=>'','image_id'=>0]; ?>
        <div class="hqui-card">
          <h4><?php echo esc_html($k);?></h4>
          <label><?php _e('Başlık','balor-quiz');?></label>
          <input type="text" name="hqui[results][<?php echo $k;?>][title]" value="<?php echo esc_attr($r['title']);?>" class="widefat">
          <label><?php _e('Açıklama','balor-quiz');?></label>
          <textarea name="hqui[results][<?php echo $k;?>][desc]" class="widefat" rows="5"><?php echo esc_textarea($r['desc']);?></textarea>

          <div class="hqui-media">
            <?php $img_id = intval($r['image_id'] ?? 0); $img_url = $img_id ? wp_get_attachment_image_url($img_id, 'medium') : ''; ?>
            <label><?php _e('Görsel (sonuç altında gösterilir)','balor-quiz');?></label>
            <div class="hqui-media-preview" style="margin:6px 0;">
              <?php if($img_url): ?>
                <img src="<?php echo esc_url($img_url); ?>" style="max-width:100%;height:auto;border:1px solid #eee;border-radius:6px;">
              <?php endif; ?>
            </div>
            <input type="hidden" class="hqui-media-id" name="hqui[results][<?php echo $k;?>][image_id]" value="<?php echo $img_id; ?>">
            <button type="button" class="button hqui-media-select"><?php _e('Görsel Seç','balor-quiz'); ?></button>
            <button type="button" class="button-link-delete hqui-media-remove"><?php _e('Kaldır','balor-quiz'); ?></button>
          </div>
        </div>
      <?php endforeach; ?>
    </div>

    <template id="hqui-q-template">
      <?php hqui_question_block('__INDEX__', ['q'=>'','opts'=>[['k'=>'A','t'=>''],['k'=>'B','t'=>''],['k'=>'C','t'=>''],['k'=>'D','t'=>'']]]); ?>
    </template>
  </div>
  <?php
}

function hqui_question_block($index, $q){
  $qtext = isset($q['q']) ? $q['q'] : '';
  $opts = isset($q['opts']) && is_array($q['opts']) ? $q['opts'] : [['k'=>'A','t'=>''],['k'=>'B','t'=>''],['k'=>'C','t'=>''],['k'=>'D','t'=>'']];
  ?>
  <div class="hqui-card hqui-q" data-index="<?php echo esc_attr($index);?>">
    <div class="hqui-q-head">
      <strong><?php _e('Soru','balor-quiz'); ?> #<span class="hqui-q-no"><?php echo is_numeric($index)? intval($index)+1 : '{{n}}'; ?></span></strong>
      <button type="button" class="button-link-delete hqui-remove-q">&times;</button>
    </div>
    <label><?php _e('Soru Metni','balor-quiz'); ?></label>
    <input type="text" name="hqui[questions][<?php echo $index; ?>][q]" value="<?php echo esc_attr($qtext);?>" class="widefat">

    <div class="hqui-opts">
      <?php 
        $letters = ['A','B','C','D'];
        foreach($letters as $i=>$letter){
          $t = isset($opts[$i]['t']) ? $opts[$i]['t'] : '';
      ?>
        <div class="hqui-opt">
          <label><?php echo esc_html($letter);?></label>
          <input type="hidden" name="hqui[questions][<?php echo $index;?>][opts][<?php echo $i;?>][k]" value="<?php echo esc_attr($letter);?>">
          <input type="text" name="hqui[questions][<?php echo $index;?>][opts][<?php echo $i;?>][t]" value="<?php echo esc_attr($t);?>" class="widefat" placeholder="<?php printf(__('Seçenek %s metni','balor-quiz'), $letter); ?>">
        </div>
      <?php } ?>
    </div>
  </div>
  <?php
}

// Save
add_action('save_post_balor_quiz_cpt', function($post_id){
  if(defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
  if(!isset($_POST['hqui_nonce']) || !wp_verify_nonce($_POST['hqui_nonce'], 'hqui_save')) return;
  if(!current_user_can('edit_post',$post_id)) return;

  $in = isset($_POST['hqui']) ? wp_unslash($_POST['hqui']) : [];
  $data = [
    'intro_title' => sanitize_text_field($in['intro_title'] ?? ''),
    'intro_desc'  => sanitize_textarea_field($in['intro_desc'] ?? ''),
    'tie_break_order' => array_values(array_filter(array_map('trim', explode(',', $in['tie_break'] ?? 'A,B,C,D')))),
    'questions' => [],
    'results' => []
  ];
  if(empty($data['tie_break_order'])) $data['tie_break_order'] = ['A','B','C','D'];

  if(!empty($in['questions']) && is_array($in['questions'])){
    foreach($in['questions'] as $q){
      $qtext = sanitize_text_field($q['q'] ?? '');
      if($qtext==='') continue;
      $opts = [];
      if(isset($q['opts']) && is_array($q['opts'])){
        foreach($q['opts'] as $op){
          $k = isset($op['k']) ? strtoupper(sanitize_text_field($op['k'])) : '';
          $t = sanitize_text_field($op['t'] ?? '');
          if(in_array($k, ['A','B','C','D']) && $t!==''){
            $opts[] = ['k'=>$k,'t'=>$t];
          }
        }
      }
      if(count($opts)===4){
        $data['questions'][] = ['q'=>$qtext, 'opts'=>$opts];
      }
    }
  }

  if(isset($in['results']) && is_array($in['results'])){
    foreach(['A','B','C','D'] as $k){
      $title = sanitize_text_field($in['results'][$k]['title'] ?? '');
      $desc  = sanitize_textarea_field($in['results'][$k]['desc'] ?? '');
      $imgid = intval($in['results'][$k]['image_id'] ?? 0);
      $data['results'][$k] = ['title'=>$title, 'desc'=>$desc, 'image_id'=>$imgid];
    }
  }

  update_post_meta($post_id, '_hqui_data', wp_json_encode($data, JSON_UNESCAPED_UNICODE));
});

// Shortcode
add_shortcode('balor_quiz', function($atts){
  $atts = shortcode_atts(['id'=>'' ], $atts, 'balor_quiz');
  $data = [];
  if(!empty($atts['id'])){
    $raw = get_post_meta(intval($atts['id']), '_hqui_data', true);
    $data = json_decode($raw,true);
  }
  if(empty($data) || empty($data['questions'])){
    return '<p>'.esc_html__('Quiz bulunamadı ya da yapılandırılmadı.','balor-quiz').'</p>';
  }
  // Add image URLs
  if (!empty($data['results']) && is_array($data['results'])){
    foreach($data['results'] as $rk => $rv){
      if(isset($rv['image_id']) && $rv['image_id']){
        $url = wp_get_attachment_image_url(intval($rv['image_id']), 'large');
        if($url){ $data['results'][$rk]['image'] = $url; }
      }
    }
  }

  wp_register_style('bq-style', BQUI_URL.'assets/css/frontend.css',[],BQUI_VER);
  wp_register_script('bq-app', BQUI_URL.'assets/js/frontend.js',[],BQUI_VER,true);
  wp_enqueue_style('bq-style');
  wp_enqueue_script('bq-app');

  ob_start(); ?>
  <div class="bq-wrap" data-quiz='<?php echo esc_attr(wp_json_encode($data, JSON_UNESCAPED_UNICODE)); ?>'>
    <div class="bq-intro">
      <h2></h2>
      <p></p>
      <button type="button" class="bq-start"><?php esc_html_e('Teste Başla','balor-quiz'); ?></button>
    </div>
    <form class="bq-form" style="display:none" aria-live="polite" novalidate>
      <div class="bq-progress" aria-hidden="true"><span class="bq-bar" style="width:0%"></span></div>
      <div class="bq-steps"></div>
      <div class="bq-nav">
        <button type="button" class="bq-prev" disabled><?php esc_html_e('Geri','balor-quiz'); ?></button>
        <button type="button" class="bq-next"><?php esc_html_e('İleri','balor-quiz'); ?></button>
        <button type="submit" class="bq-submit" style="display:none"><?php esc_html_e('Sonucu Göster','balor-quiz'); ?></button>
      </div>
    </form>
    <div class="bq-result" style="display:none"></div>
  </div>
  <?php
  return ob_get_clean();
});

/** Shortcode meta box */
add_action('add_meta_boxes', function () {
  add_meta_box(
    'bqui_shortcode',
    __('Shortcode', 'balor-quiz'),
    function ($post) {
      if ($post->post_type !== 'balor_quiz_cpt') return;
      $id = (int) $post->ID;
      $short = '[balor_quiz id="' . $id . '"]';
      ?>
      <p><?php _e('Bu quizi eklemek için kısa kod:', 'balor-quiz'); ?></p>
      <input type="text" class="widefat bqui-shortcode-input" readonly value="<?php echo esc_attr($short); ?>">
      <p><button type="button" class="button" id="bqui-copy"><?php _e('Kopyala','balor-quiz'); ?></button></p>
      <script>
        (function(){
          const btn = document.getElementById('bqui-copy');
          const inp = document.querySelector('.bqui-shortcode-input');
          btn?.addEventListener('click', function(){
            inp.select(); document.execCommand('copy');
            btn.textContent = '<?php echo esc_js(__('Kopyalandı!','balor-quiz')); ?>';
            setTimeout(()=> btn.textContent = '<?php echo esc_js(__('Kopyala','balor-quiz')); ?>', 1500);
          });
        })();
      </script>
      <style>.bqui-shortcode-input{font-family:monospace}</style>
      <?php
    },
    'balor_quiz_cpt',
    'side',
    'high'
  );
});

/** Shortcode column in list */
add_filter('manage_balor_quiz_cpt_posts_columns', function($cols){
  $cols['bqui_shortcode'] = __('Shortcode','balor-quiz');
  return $cols;
});
add_action('manage_balor_quiz_cpt_posts_custom_column', function($col, $post_id){
  if ($col === 'bqui_shortcode') {
    echo '<code>[balor_quiz id="'.(int)$post_id.'"]</code>';
  }
}, 10, 2);
